
interface ProcessedResponse<T = any> {
    status: number,
    json: T | null,
    text: string
}
type AssetType = "Dataset" | "Model" | "Documentation" | "Application" | "Web - API" | "Web-based tool" | "Client" | "Library"
type AccessType = "PUBLIC" | "RESTRICTED" | "CONFIDENTIAL"
interface AssetPolicy {
    assetId: string,
    assetType: AssetType,
    accessType: AccessType,
    rule?: string

}
type AssetControllerEndpointType = "access" | "visibility"
export interface PolicyIntegrationServiceConstructor {
    apmURL: string,
    jwtToken?: string,

}

export class PolicyIntegrationService {
    private apmURL: string
    private jwtToken: string | undefined

    constructor({ apmURL, jwtToken }: PolicyIntegrationServiceConstructor) {

        this.jwtToken = jwtToken
        this.apmURL = apmURL
    }
    async checkId(id: string, type: string): Promise<boolean> {
        return true
    }




    private getURL(path: string) {
        return `${this.apmURL}${path}`
    }

    private getAuthorizationHeader() {
        return {
            Authorization: `${this.jwtToken || ""}`
        }
    }

    private async processResponse<T = any>(response: Response): Promise<ProcessedResponse<T>> {
        let json: T | null = null
        const text = await response.text()
        try {
            json = JSON.parse(text)
        } catch (err: any) {
            console.warn("Failed to obrain JSON", err)
        }

        return {
            status: response.status,
            json,
            text
        }
    }

    assetPolicyEditor = {
        get: async (assetId: string) => {
            const headers = this.getAuthorizationHeader()
            if (!headers) return null
            const url = this.getURL(`/public/api/v1.0/asset-policy-editor?assetId=${assetId}`)
            const response = await fetch(url, { headers })
            return this.processResponse(response)

        },
        put: async (request: AssetPolicy) => {
            const headers = this.getAuthorizationHeader()
            if (!headers) return null
            const url = this.getURL("/public/api/v1.0/asset-policy-editor")
            const body = JSON.stringify(request)
            const response = await fetch(url,
                {
                    method: "PUT",
                    headers: {
                        ...headers,
                        "Content-Type": "application/json"
                    },
                    body
                }
            )
            return this.processResponse(response)
        },
        post: async (request: AssetPolicy) => {
            const headers = this.getAuthorizationHeader()
            if (!headers) return null
            const url = this.getURL("/private/api/v1.0/asset-policy-editor")
            const body = JSON.stringify(request)
            const response = await fetch(url,
                {
                    method: "POST",
                    headers: {
                        ...headers,
                        "Content-Type": "application/json"
                    },
                    body
                }
            )
            return this.processResponse(response)
        },
        delete: async (assetId: string) => {
            const headers = this.getAuthorizationHeader()
            if (!headers) return null
            const url = this.getURL(`/private/api/v1.0/asset-policy-editor?assetId=${assetId}`)
            const response = await fetch(url, { method: "DELETE", headers })
            return this.processResponse(response)
        }
    }
    private assetController = {
        checkMany: async (key: AssetControllerEndpointType, request: string[]) => {
            const headers = this.getAuthorizationHeader()
            if (!headers) return null
            const url = this.getURL(`/public/api/v1.0/asset-${key}/check-many`)
            const body = JSON.stringify(request)
            const response = await fetch(url, {
                method: "POST",
                body,
                headers: {
                    ...headers,
                    "Content-Type": "application/json"
                }
            })
            return this.processResponse(response)
        },
        retrieveAll: async (key: AssetControllerEndpointType, assetType?: AssetType) => {
            const headers = this.getAuthorizationHeader()
            if (!headers) return null
            let url = this.getURL(`/public/api/v1.0/asset-${key}/retrieve-all`)
            if (assetType) url = `${url}?assetType=${assetType}`
            const response = await fetch(url, { headers })
            return this.processResponse(response)
        },
        checkOne: async (key: AssetControllerEndpointType, assetId: string) => {
            const headers = this.getAuthorizationHeader()
            if (!headers) return null
            const url = this.getURL(`/public/api/v1.0/asset-${key}/check-one?assetId=${assetId}`)
            const response = await fetch(url, { headers })
            return this.processResponse(response)
        }

    }

    assetContentAccessController = {
        checkMany: (request: string[]) => { return this.assetController.checkMany("access", request) },
        retrieveAll: (assetType?: AssetType) => { return this.assetController.retrieveAll("access", assetType) },
        checkOne: (assetId: string) => { return this.assetController.checkOne("access", assetId) }

    }

    assetMarketPlaceVisibilityController = {
        checkMany: (request: string[]) => { return this.assetController.checkMany("visibility", request) },
        retrieveAll: (assetType?: AssetType) => { return this.assetController.retrieveAll("visibility", assetType) },
        checkOne: (assetId: string) => { return this.assetController.checkOne("visibility", assetId) }

    }
}